<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

/**
 * CoinEx API v2 Service
 * Documentation: https://docs.coinex.com/api/v2
 */
class CoinExService
{
    protected $apiKey;
    protected $apiSecret;
    protected $baseUrl = 'https://api.coinex.com/v2';
    
    public function __construct()
    {
        $this->apiKey = getSetting('coinex_api_key');
        $this->apiSecret = getSetting('coinex_api_secret');
    }
    
    /**
     * Get all available markets
     */
    public function getMarkets()
    {
        try {
            $response = $this->publicRequest('GET', '/spot/market');
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return [];
        } catch (\Exception $e) {
            Log::error('CoinEx getMarkets error: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get ticker for a specific market
     */
    public function getTicker($market)
    {
        try {
            $response = $this->publicRequest('GET', '/spot/ticker', ['market' => $market]);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return null;
        } catch (\Exception $e) {
            Log::error('CoinEx getTicker error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get order book (depth)
     */
    public function getOrderBook($market, $limit = 20, $interval = '0')
    {
        try {
            $response = $this->publicRequest('GET', '/spot/depth', [
                'market' => $market,
                'limit' => $limit,
                'interval' => $interval
            ]);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return null;
        } catch (\Exception $e) {
            Log::error('CoinEx getOrderBook error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get account balances
     */
    public function getBalances()
    {
        try {
            $response = $this->privateRequest('GET', '/assets/spot/balance');
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return [];
        } catch (\Exception $e) {
            Log::error('CoinEx getBalances error: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Place a limit order
     */
    public function placeLimitOrder($market, $side, $amount, $price)
    {
        try {
            $params = [
                'market' => $market,
                'side' => $side, // 'buy' or 'sell'
                'type' => 'limit',
                'amount' => (string)$amount,
                'price' => (string)$price,
            ];
            
            $response = $this->privateRequest('POST', '/spot/order', $params);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return null;
        } catch (\Exception $e) {
            Log::error('CoinEx placeLimitOrder error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Place a market order
     */
    public function placeMarketOrder($market, $side, $amount)
    {
        try {
            $params = [
                'market' => $market,
                'side' => $side,
                'type' => 'market',
                'amount' => (string)$amount,
            ];
            
            $response = $this->privateRequest('POST', '/spot/order', $params);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return null;
        } catch (\Exception $e) {
            Log::error('CoinEx placeMarketOrder error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Cancel an order
     */
    public function cancelOrder($market, $orderId)
    {
        try {
            $params = [
                'market' => $market,
                'order_id' => $orderId,
            ];
            
            $response = $this->privateRequest('POST', '/spot/cancel-order', $params);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return null;
        } catch (\Exception $e) {
            Log::error('CoinEx cancelOrder error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Get pending orders
     */
    public function getPendingOrders($market = null)
    {
        try {
            $params = [];
            if ($market) {
                $params['market'] = $market;
            }
            
            $response = $this->privateRequest('GET', '/spot/pending-order', $params);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return [];
        } catch (\Exception $e) {
            Log::error('CoinEx getPendingOrders error: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get finished orders
     */
    public function getFinishedOrders($market, $page = 1, $limit = 100)
    {
        try {
            $params = [
                'market' => $market,
                'page' => $page,
                'limit' => $limit,
            ];
            
            $response = $this->privateRequest('GET', '/spot/finished-order', $params);
            
            if ($response && isset($response['data'])) {
                return $response['data'];
            }
            
            return [];
        } catch (\Exception $e) {
            Log::error('CoinEx getFinishedOrders error: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Public API request (no authentication)
     */
    protected function publicRequest($method, $endpoint, $params = [])
    {
        $url = $this->baseUrl . $endpoint;
        
        if ($method === 'GET' && !empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'User-Agent: Mozilla/5.0'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            Log::error("CoinEx API error: HTTP $httpCode - $response");
            return null;
        }
        
        return json_decode($response, true);
    }
    
    /**
     * Private API request (requires authentication)
     */
    protected function privateRequest($method, $endpoint, $params = [])
    {
        if (!$this->apiKey || !$this->apiSecret) {
            throw new \Exception('CoinEx API credentials not configured');
        }
        
        $timestamp = time() * 1000; // milliseconds
        $preparedParams = $this->prepareParams($params);
        
        // Create signature
        $signStr = $method . $endpoint . $preparedParams . $timestamp;
        $signature = hash_hmac('sha256', $signStr, $this->apiSecret);
        
        $url = $this->baseUrl . $endpoint;
        
        $headers = [
            'Content-Type: application/json',
            'X-COINEX-KEY: ' . $this->apiKey,
            'X-COINEX-SIGN: ' . $signature,
            'X-COINEX-TIMESTAMP: ' . $timestamp,
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $preparedParams);
        } elseif ($method === 'GET' && !empty($params)) {
            curl_setopt($ch, CURLOPT_URL, $url . '?' . http_build_query($params));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            Log::error("CoinEx Private API error: HTTP $httpCode - $response");
            throw new \Exception("CoinEx API returned error: $response");
        }
        
        return json_decode($response, true);
    }
    
    /**
     * Prepare parameters for signature
     */
    protected function prepareParams($params)
    {
        if (empty($params)) {
            return '';
        }
        
        ksort($params);
        return json_encode($params, JSON_UNESCAPED_SLASHES);
    }
}
