<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Cryptocurrency;
use App\functions;
use Illuminate\Http\Request;
use Auth;

class LandingController extends Controller
{
    /**
     * Display the landing page
     */
    public function index()
    {
        // Show landing page to everyone (no redirect for authenticated users)
        return view('landing');
    }

    /**
     * Get live cryptocurrency prices for the landing page
     * Public endpoint - no authentication required
     */
    public function getCryptoPrices()
    {
        try {
            $priceService = new \App\Services\CoinPriceService();
            $allPrices = $priceService->getAllPrices();
            
            \Log::info('LandingController: Fetched prices', ['count' => count($allPrices)]);
            
            $cryptocurrencies = Cryptocurrency::all();
            $prices = [];

            foreach ($cryptocurrencies as $crypto) {
                try {
                    $symbol = $crypto->symbol;
                    $priceData = null;
                    
                    // Get price data from service
                    if (isset($allPrices[$symbol])) {
                        $priceData = $allPrices[$symbol];
                    } else {
                        \Log::warning('LandingController: Price not found for ' . $symbol);
                    }
                    
                    $prices[] = [
                        'name' => $crypto->name,
                        'name_fa' => $crypto->name_fa,
                        'symbol' => $symbol,
                        'price_usdt' => $priceData['usdt_price'] ?? 0,
                        'price_irt' => isset($priceData['buy']) ? number_format($priceData['buy'], 0, '.', ',') : '0',
                        'change_24h' => isset($priceData['change_24h']) ? number_format($priceData['change_24h'], 2) : '0.00',
                        'stock' => $crypto->stock ?? 0,
                    ];

                } catch (\Exception $e) {
                    \Log::error('LandingController: Error processing crypto', [
                        'symbol' => $symbol ?? 'unknown',
                        'error' => $e->getMessage()
                    ]);
                    continue;
                }
            }

            return response()->json([
                'status' => 'success',
                'data' => $prices,
                'debug' => [
                    'total_cryptos' => count($cryptocurrencies),
                    'prices_fetched' => count($allPrices),
                    'prices_returned' => count($prices)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('LandingController: Failed to fetch prices', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch cryptocurrency prices: ' . $e->getMessage(),
                'data' => []
            ], 500);
        }
    }
    
    /**
     * Preview landing page (for admins)
     */
    public function preview()
    {
        return view('landing');
    }
}
