<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\UserWallet;
use App\Services\CoinExService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CryptoWalletController extends Controller
{
    protected $coinex;

    public function __construct()
    {
        $this->middleware('auth');
        $this->coinex = new CoinExService();
    }

    /**
     * Show crypto wallet page
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get all user crypto wallets
        $wallets = UserWallet::where('user_id', $user->id)
            ->orderBy('balance', 'desc')
            ->get();

        // Calculate total in USDT
        $totalUSDT = 0;
        foreach ($wallets as $wallet) {
            if ($wallet->currency === 'USDT') {
                $totalUSDT += $wallet->balance;
            } else {
                // Get price from CoinEx
                try {
                    $ticker = $this->coinex->getTicker($wallet->currency . 'USDT');
                    if ($ticker && isset($ticker['last'])) {
                        $totalUSDT += $wallet->balance * $ticker['last'];
                    }
                } catch (\Exception $e) {
                    // Skip if market not found
                }
            }
        }

        return view('user.crypto-wallet.index', compact('wallets', 'totalUSDT'));
    }

    /**
     * Get user balances (API)
     */
    public function getBalances()
    {
        $user = Auth::user();
        
        $wallets = UserWallet::where('user_id', $user->id)->get();

        return response()->json([
            'status' => true,
            'data' => $wallets
        ]);
    }

    /**
     * Get balance for specific currency
     */
    public function getBalance($currency)
    {
        $user = Auth::user();
        
        $wallet = UserWallet::where('user_id', $user->id)
            ->where('currency', $currency)
            ->first();

        if (!$wallet) {
            $wallet = UserWallet::create([
                'user_id' => $user->id,
                'currency' => $currency,
                'balance' => 0,
                'locked_balance' => 0,
            ]);
        }

        return response()->json([
            'status' => true,
            'data' => $wallet
        ]);
    }

    /**
     * Sync balances from CoinEx
     */
    public function syncBalances()
    {
        $user = Auth::user();

        try {
            // Get balances from CoinEx
            $coinexBalances = $this->coinex->getBalances();

            if (empty($coinexBalances)) {
                return response()->json([
                    'status' => false,
                    'message' => 'خطا در دریافت موجودی از صرافی'
                ], 400);
            }

            DB::beginTransaction();

            foreach ($coinexBalances as $currency => $balance) {
                if (isset($balance['available']) && $balance['available'] > 0) {
                    $wallet = UserWallet::firstOrCreate([
                        'user_id' => $user->id,
                        'currency' => $currency
                    ]);

                    $wallet->balance = $balance['available'];
                    $wallet->locked_balance = $balance['frozen'] ?? 0;
                    $wallet->save();
                }
            }

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'موجودی با موفقیت همگام‌سازی شد'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'status' => false,
                'message' => 'خطا در همگام‌سازی: ' . $e->getMessage()
            ], 400);
        }
    }
}
