<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use App\TradingPair;

class TradingDataController extends Controller
{
    /**
     * Get real-time ticker for a market
     */
    public function getTicker($market)
    {
        $ticker = Cache::get("ticker:$market");

        if (!$ticker) {
            return response()->json([
                'status' => false,
                'message' => 'No data available'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $ticker
        ]);
    }

    /**
     * Get real-time order book for a market
     */
    public function getOrderBook($market)
    {
        $orderBook = Cache::get("orderbook:$market");

        if (!$orderBook) {
            return response()->json([
                'status' => false,
                'message' => 'No data available'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $orderBook
        ]);
    }

    /**
     * Get all tickers
     */
    public function getAllTickers()
    {
        $tradingPairs = TradingPair::where('is_active', true)->get();
        $tickers = [];

        foreach ($tradingPairs as $pair) {
            $ticker = Cache::get("ticker:{$pair->market}");
            if ($ticker) {
                $tickers[] = $ticker;
            }
        }

        return response()->json([
            'status' => true,
            'data' => $tickers
        ]);
    }

    /**
     * Get market summary (for charts)
     */
    public function getMarketSummary($market)
    {
        $ticker = Cache::get("ticker:$market");
        $orderBook = Cache::get("orderbook:$market");

        if (!$ticker) {
            return response()->json([
                'status' => false,
                'message' => 'No data available'
            ], 404);
        }

        $summary = [
            'market' => $market,
            'price' => $ticker['last'],
            'change_24h' => $ticker['change'] ?? 0,
            'high_24h' => $ticker['high'] ?? 0,
            'low_24h' => $ticker['low'] ?? 0,
            'volume_24h' => $ticker['volume'] ?? 0,
            'best_bid' => isset($orderBook['bids'][0]) ? $orderBook['bids'][0][0] : null,
            'best_ask' => isset($orderBook['asks'][0]) ? $orderBook['asks'][0][0] : null,
            'updated_at' => $ticker['updated_at'] ?? time(),
        ];

        return response()->json([
            'status' => true,
            'data' => $summary
        ]);
    }

    /**
     * SSE (Server-Sent Events) endpoint for real-time updates
     */
    public function streamTicker($market)
    {
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no');

        $lastUpdate = 0;

        while (true) {
            $ticker = Cache::get("ticker:$market");

            if ($ticker && ($ticker['updated_at'] ?? 0) > $lastUpdate) {
                echo "data: " . json_encode($ticker) . "\n\n";
                ob_flush();
                flush();
                $lastUpdate = $ticker['updated_at'];
            }

            sleep(1);

            // Check if client disconnected
            if (connection_aborted()) {
                break;
            }
        }
    }
}
